/*
 * Copyright (c) 2004 ONStor, Inc.
 *
 * This file provides routines for mapping marvel memory into the
 * kernel's virtual address space.
 */

#include <linux/kernel.h>
#include <linux/types.h>
#include <asm/page.h>
#include <asm/pgtable-bits.h>
#include <asm/mipsregs.h>
#include <asm/fixmap.h>
#include <asm/tlbflush.h>
#include <linux/errno.h>
#include <asm/debug.h>

extern void prom_printf(char *fmt, ...);
phys_t marvelvatopa(phys_t va);
phys_t marvelpatova(phys_t pa);

#define MVMEM_PA_START	0x40000000
#define MVMEM_PA_SIZE	32*1024*1024

#define roundup(x, y) ((((x) + (y) - 1)) & ~((y) - 1))

/* physical memory address assigned by prom. */
phys_t marvel_memstart = MVMEM_PA_START;

/* size should assigned by prom. */
u_int marvel_memsize = MVMEM_PA_SIZE;

/* kernel virtual address assigned by marvel_mem_wire(). */
phys_t marvel_va;

/*
 * marvel reserved memory map.
 */ 
struct _mvmemregion {
	int pa_start;
	int pa_end;
} mvmemregion = {MVMEM_PA_START, MVMEM_PA_START + MVMEM_PA_SIZE};

static inline unsigned long ENTRYLO(unsigned long paddr)
{
	return (pte_val(pfn_pte(
	    (paddr >> PAGE_SHIFT), PAGE_KERNEL_UNCACHED))>>6 | 1);
}

/*
 * Map memory assigned to the marvel 64440 chip at a known physical address
 * to a known virtual address. We steal 32MB from the top of virtual memory.
 * The marvell memory region can not be larger than 32MB unless more virtual
 * memory is reserved.
 *
 *	Physical Start	Virtual Start
 *	0x4000.0000	(FIXADDR_TOP - 32MB)
 */
void marvell_mem_wire(void)
{
#ifdef CONFIG_32BIT
	phys_t pa, va, tva;
	int seg, size;

	write_c0_wired(0);
	local_flush_tlb_all();

	pa = MVMEM_PA_START;
	tva = fix_to_virt(__mv64x40_start);
	size = 0x20 << 20 /* 32 MB */;
	va = roundup(tva, size);
prom_printf("marvell_mem_wire: tva = %x va = %x (%x) size = %x (%x)\n", tva, va, __mv64x40_start, size, size-1);
	/*
	 * The physical address has to be "size" aligned which is
	 * the size of the wired memory segment. entrlo1 is:
	 * paddr + (size << 1)
	 */
	db_assert((pa & (size - 1)) == 0 &&  (va & (size - 1)) == 0);
	switch(size >> 20) {
		case 512: /* 512MB */
			seg = PM_256M;
			break;
		case 128: /* 128MB */
			seg = PM_64M;
			break;
		case 32: /* 32MB */
			seg = PM_16M;
			break;
		case 8: /* 8MB */
			seg = PM_4M;
			break;
		default:
			prom_printf("marvel_mem_wire() failed\n");
			return;
	}
	add_wired_entry(ENTRYLO(pa), ENTRYLO((pa + (size >> 1))), va, seg);
prom_printf("marvell_mem_wire: size = %x, pa_l0 = %x, pa_l1 = %x, seg = %x\n", size, ENTRYLO(pa), ENTRYLO((pa + (size>>1))), seg);
	marvel_va = va;
	dump_tlb(0, 63);
#endif
}

/*
 * Allocate a range of physically contigous memory from the marvel memory
 * region. The size of the requested region must be page aligned.
 */
phys_t
marvel_mem_alloc(int size, int *err)
{
	phys_t pa;

	if (size & (PAGE_SIZE - 1)) {
		if (err)
			*err = EINVAL;
		return (-1);
	}
	
	pa = mvmemregion.pa_start;
	if ((pa + size) > mvmemregion.pa_end) {
		*err = ENOMEM;
		return (-1);
	}
	mvmemregion.pa_start = pa + size;	

prom_printf("marvel_mem_alloc: pa = %x, va = %x\n", pa, marvelpatova(pa));
	return (pa);
}

/*
 * free an allocate region of marvel memory.
 */
int
marvel_mem_free(phys_t pa, int size)
{
	/* currently not supported. */
	return (0);
}

/*
 * convert a marvel virtual memory address to its physical memory
 * address.
 */
phys_t
marvelvatopa(phys_t va)
{
	db_assert(va >= marvel_va && va <= (marvel_va + marvel_memsize));
	return (marvel_memstart + (va - marvel_va));
}

/*
 * convert a marvel physical memory address to its virtual memory
 * address.
 */
phys_t
marvelpatova(phys_t pa)
{
	db_assert(pa >= marvel_memstart && pa <= (marvel_memstart
	     + marvel_memsize));
	return (marvel_va + (pa - marvel_memstart));
}

/*
 * Verify that virtual address is from the Marvel memory region.
 */
int
ismarvelmem(phys_t va)
{
	return ((va >= marvel_va && va <= marvel_va + marvel_memsize));
}
