/*
 * Setup the MV64x40 serial port interface to work in DMA mode and
 * debug port mode which is primarily used during early boot, and for
 * KGDB.
 *
 * Copyright (C) 2005 Onstor, Inc.
 */
#include <linux/init.h>
#include <linux/console.h>
#include <linux/kdev_t.h>
#include <linux/major.h>
#include <linux/termios.h>
#include <linux/sched.h>
#include <linux/tty.h>

#include <linux/serial.h>
#include <linux/serial_core.h>
#include <asm/serial.h>
#include <asm/io.h>
#include "mv64x40.h"

#define BOBCAT_DEFAULT_BAUD	57600
#define BOBCAT_MPSC_CLK_SRC	0x8	/* clock src is system clock. */

#define MPSC_CHR_5_V	(1<<15)
#define MPSC_CHR_4_Z	(1<<29)
#define MPSC_CHR_2_TCS	(1<<9)
#define MPSC_CHR_10_RCRn_mask (0xff << 16)

#define MPSC_CHR_1 	0x000c
#define MPSC_CHR_2 	0x0010
#define MPSC_CHR_3 	0x0014
#define MPSC_CHR_4 	0x0018
#define MPSC_CHR_5 	0x001c
#define MPSC_CHR_6 	0x0020
#define MPSC_CHR_7 	0x0024
#define MPSC_CHR_8 	0x0028
#define MPSC_CHR_9 	0x002c
#define MPSC_CHR_10	0x0030

char *bobcat_cons_base = (char *)(CKSEG1ADDR(MV64x40_MPSC_0_ADDR));
int bobcatconstimedout = 0;

#ifdef BOBCAT_PLATFORM_DEVICE_FIXUP
/*
 * The platform_add_device() interface has a callback function named
 * platform_notify() that is used to do platform specific fixups to
 * the device being added. To enable the callback, set
 *	platform_notify = bobcat_platform_notify;
 */
static void __init
bobcat_fixup_mpsc_data(struct platform_data *pdev)
{
	struct mpsc_pdata *pdata;

	pdata = (struct mpsc_pdata *)pdev->dev.platform.data;

	pdata->max_idle = 40;
	pdata->default_baud = BOBCAT_DEFAULT_BAUD;
	pdata->brg_clk_src = BOBCAT_MPSC_CLK_SRC;
	pdata->brg_clk_freq = bus_clock;
}

static int __init
bobcat_platform_notify(struct device *dev)
{
	static struct {
		char *bus_id;
		void *((*rtn)(struct platform_device *pdev));
	} dev_map[] = {
		{MPSC_CTLR_NAME ".0", bobcat_fixup_mpsc_pdata}	
		{MPSC_CTLR_NAME ".1", bobcat_fixup_mpsc_pdata}	
	};
	struct platform_device *pdev;
	int i;

	if (dev && dev->bus_id) {
		for (i = 0; i < ARRAY_SIZE(dev_map); i++) {
			if (!strncmp(dev->bus_id, dev_map[i].bus_id,
			    BUS_ID_SIZE)) {
				pdev = container_of(dev,
				    struct platform_device, dev);
				dev_map[i].rtn(pdev);
			}
		}
	}
	return (0);
}
#endif

/*
 * Turn the debug port capability on MPSC port 0.
 */
void DebugPortOn(void)
{
	/*
	 * Set the "Z" bit in MPSC_CHR_4 register to enable
	 * writing one character at a time via the TCS (transmit
	 * character to send) field in register CHR_1.
	 */
	writel(MPSC_CHR_4_Z, bobcat_cons_base + MPSC_CHR_4);
	/*
	 * Set the "V" bit in CHR_5 to enable receiving one byte
	 * at a time via the RCRn field in  CHR_10.
	 */
	writel(MPSC_CHR_5_V, bobcat_cons_base + MPSC_CHR_5);
}

/*
 * Turn off the debug port uart capability.
 */
void DebugPortOff(void)
{
	/*
	 * clear the "Z" bit in MPSC_CHR_4 register, and
	 * clear the "V" bit in MPSC_CHR_5 register.
	 */
	writel(0, bobcat_cons_base + MPSC_CHR_4);
	writel(0, bobcat_cons_base + MPSC_CHR_5);
}

/*
 * KGDB interface to write one character.
 */
int
putDebugChar(char c)
{
	unsigned long v;
	int delaycnt = 1000;

	writel((unsigned char)c, bobcat_cons_base + MPSC_CHR_1);
	v = readl(bobcat_cons_base + MPSC_CHR_2);
	writel(v | MPSC_CHR_2_TCS, bobcat_cons_base + MPSC_CHR_2);
	while (readl(bobcat_cons_base + MPSC_CHR_2) & MPSC_CHR_2_TCS) {
		if (delaycnt-- < 0) {
			bobcatconstimedout++;
			break;
		}
	}
	return (1);
}

/*
 * KGDB interface to read one character.
 */
char
getDebugChar(void)
{
	unsigned long v;
	char c;

	while (((v = readl(bobcat_cons_base + MPSC_CHR_10))
	     & MPSC_CHR_10_RCRn_mask) == 0);
	/* clear the received character in MPSC_CHR_10. */
	writel((v | MPSC_CHR_10_RCRn_mask), bobcat_cons_base + MPSC_CHR_10);
	
	c = (v & MPSC_CHR_10_RCRn_mask) >> 16;
	return (c);
}

void prom_putchar(char c)
{
	putDebugChar(c);
}

char __init prom_getchar(void)
{
	return 0;
}
