/*
 * Copyright (C) 2000, 2001, 2002, 2003 Broadcom Corporation
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* 
 * Driver support for the on-chip sb1250 dual-channel serial port,
 * running in asynchronous mode.  Also, support for doing a serial console
 * on one of those ports 
 */
#include <linux/config.h>
#include <linux/types.h>
#include <linux/serial.h>
#include <linux/module.h>
#include <linux/console.h>
#include <linux/kdev_t.h>
#include <linux/major.h>
#include <linux/termios.h>
#include <linux/spinlock.h>
#include <linux/irq.h>
#include <linux/errno.h>
#include <linux/tty.h>
#include <linux/sched.h>
#include <linux/tty_flip.h>
#include <linux/timer.h>
#include <linux/init.h>
#include <linux/mm.h>
#include <asm/delay.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <linux/serial_core.h>
#include <linux/serial.h>
#include <asm/sibyte/swarm.h>
#include <asm/sibyte/sb1250_regs.h>
#include <asm/sibyte/sb1250_uart.h>
#include <asm/sibyte/sb1250_int.h>
#include <asm/sibyte/sb1250.h>
#include <asm/sibyte/64bit.h>
#include <asm/war.h>

#define IO_SPACE_BASE	KSEG1

/* Toggle spewing of debugging output */
#undef DUART_SPEW

#define DEFAULT_CFLAGS          (CS8 | B115200)

#define TX_INTEN          1
#define DUART_INITIALIZED 2

#ifndef MIN
#define MIN(a,b)	((a) < (b) ? (a) : (b))
#endif

#define DUART_MAX_LINE 2
char sb1250_duart_present[DUART_MAX_LINE] = {1,1};

typedef struct { 
	struct uart_port	port;
	unsigned int	last_cflags;
	/* CSR addresses */
	u32		*status;
	u32		*imr;
	u32		*tx_hold;
	u32		*rx_hold;
	u32		*mode_1;
	u32		*mode_2;
	u32		*clk_sel;
	u32		*cmd;
} duart_port_t;

static duart_port_t duart_port[DUART_MAX_LINE];

static void duart_stop_tx(struct uart_port *port, unsigned int tty_stop);

/*
 * In bug 1956, we get glitches that can mess up uart registers.  This
 * "write-mode-1 after any register access" is the accepted
 * workaround.
 */
#if SIBYTE_1956_WAR
static unsigned int last_mode1[DUART_MAX_LINE];
#endif

static inline u32 READ_SERCSR(u32 *addr, int line)
{
	u32 val = csr_in32(addr);
#if SIBYTE_1956_WAR
	csr_out32(last_mode1[line], duart_port[line].mode_1);
#endif
	return val;
}

static inline void WRITE_SERCSR(u32 val, u32 *addr, int line)
{
	csr_out32(val, addr);
#if SIBYTE_1956_WAR
	csr_out32(last_mode1[line], duart_port[line].mode_1);
#endif
}

static void init_duart_port(duart_port_t *up, int line)
{
	up->status = (u32 *)(IO_SPACE_BASE |
	    A_DUART_CHANREG(line, R_DUART_STATUS));
	up->imr = (u32 *)(IO_SPACE_BASE | A_DUART_IMRREG(line));
	up->tx_hold = (u32 *)(IO_SPACE_BASE |
	    A_DUART_CHANREG(line, R_DUART_TX_HOLD));
	up->rx_hold = (u32 *)(IO_SPACE_BASE |
	    A_DUART_CHANREG(line, R_DUART_RX_HOLD));
	up->mode_1 = (u32 *)(IO_SPACE_BASE |
	    A_DUART_CHANREG(line, R_DUART_MODE_REG_1));
	up->mode_2 = (u32 *)(IO_SPACE_BASE |
	    A_DUART_CHANREG(line, R_DUART_MODE_REG_2));
	up->clk_sel = (u32 *)(IO_SPACE_BASE |
	    A_DUART_CHANREG(line, R_DUART_CLK_SEL));
	up->cmd = (u32 *)(IO_SPACE_BASE |
	    A_DUART_CHANREG(line, R_DUART_CMD));
}

/*
 * Mask out the passed interrupt lines at the duart level.  This should be
 * called while holding the associated outp_lock.
 */
static inline void duart_mask_ints(unsigned int line, unsigned int mask)
{
	duart_port_t *port = duart_port + line;
	u64 tmp = READ_SERCSR(port->imr, line);
	WRITE_SERCSR(tmp & ~mask, port->imr, line);
}

	
/* Unmask the passed interrupt lines at the duart level */
static inline void duart_unmask_ints(unsigned int line, unsigned int mask)
{
	duart_port_t *port = duart_port + line;
	u64 tmp = READ_SERCSR(port->imr, line);
	WRITE_SERCSR(tmp | mask, port->imr, line);
}

static void duart_rx_char(struct uart_port *port, struct pt_regs *regs,
    unsigned int status)
{
	duart_port_t *up = (duart_port_t *)port;
	int line = up->port.line;
	struct tty_struct *tty = up->port.info->tty;
	unsigned int ch, flag;

	if (tty->flip.count >= TTY_FLIPBUF_SIZE) {
		if (tty->low_latency)
			tty_flip_buffer_push(tty);
	}		
	ch = READ_SERCSR(up->rx_hold, line);
	flag = TTY_NORMAL;
	port->icount.rx++;
	if (status & M_DUART_OVRUN_ERR) {
		port->icount.overrun++;
		tty_insert_flip_char(tty, 0, TTY_OVERRUN);
	}
	if (status & M_DUART_PARITY_ERR) {
		port->icount.parity++;
		flag = TTY_PARITY;
	} else if (status & M_DUART_FRM_ERR) {
		port->icount.frame++;
		flag = TTY_FRAME;
	}
	if (uart_handle_sysrq_char(&up->port, ch, regs))
		return;
	tty_insert_flip_char(tty, ch, flag);
}

static inline void duart_tx_chars(struct uart_port *port, struct pt_regs* regs,
    unsigned int status)
{
	duart_port_t *up = (duart_port_t *)port;
	struct circ_buf *xmit = &up->port.info->xmit;
	int line = up->port.line;
	int count;

	if (uart_circ_empty(xmit) || uart_tx_stopped (&up->port)) {
		duart_stop_tx(&up->port, 0);
		return;
	}

	count = up->port.fifosize;
	do {
		if (!(READ_SERCSR(up->status, line) & M_DUART_TX_RDY))
			break;
		WRITE_SERCSR(xmit->buf[xmit->tail], up->tx_hold, line);
		xmit->tail = (xmit->tail + 1) & (UART_XMIT_SIZE - 1);
		up->port.icount.tx++;
		if (uart_circ_empty(xmit))
			break;
		udelay(10);
	} while (--count > 0);

	if (uart_circ_chars_pending(xmit) < WAKEUP_CHARS)
		uart_write_wakeup(&up->port);

	if (uart_circ_empty(xmit))
		duart_stop_tx(&up->port, 0);
}

/* 
 * Generic interrupt handler for both channels.  dev_id is a pointer
 * to the proper duart_port structure, so from that we can derive 
 * which port interrupted 
 */

static irqreturn_t duart_int(int irq, void *dev_id, struct pt_regs *regs)
{
	duart_port_t *up = (duart_port_t *)dev_id;
	struct tty_struct *tty = up->port.info->tty;
	unsigned int status;
	int rcv = 0;
	int loopcnt = 2048;
	int line = up->port.line;
	int handled = 0;

	spin_lock(&up->port.lock);

	status = READ_SERCSR(up->status, line);
	if (status) {
		do {
			if (status & M_DUART_RX_RDY) {
				duart_rx_char(&up->port, regs, status);
				rcv++;
			}
			if (status & M_DUART_TX_RDY) {
				duart_tx_chars(&up->port, regs, status);
			}
			if (loopcnt-- == 0)
				break;
		} while ((status = READ_SERCSR(up->status, line)) != 0);
		handled = 1;
	}
	if (rcv)
		tty_flip_buffer_push(tty);
	
	spin_unlock(&up->port.lock);
	return IRQ_RETVAL(handled);
}


/* See sb1250 user manual for details on these registers */
static inline void duart_set_cflag(unsigned int line, unsigned int cflag)
{
	unsigned int mode_reg1 = 0, mode_reg2 = 0;
	unsigned int clk_divisor;
	duart_port_t *up = duart_port + line;
	unsigned long flags;

	switch (cflag & CSIZE) {
	case CS7:
		mode_reg1 |= V_DUART_BITS_PER_CHAR_7;
		
	default:
		/* We don't handle CS5 or CS6...is there a way we're supposed to flag this? 
		   right now we just force them to CS8 */
		mode_reg1 |= 0x0;
		break;
	}
	if (cflag & CSTOPB) {
	        mode_reg2 |= M_DUART_STOP_BIT_LEN_2;
	}
	if (!(cflag & PARENB)) {
	        mode_reg1 |= V_DUART_PARITY_MODE_NONE;
	}
	if (cflag & PARODD) {
		mode_reg1 |= M_DUART_PARITY_TYPE_ODD;
	}
	
	/* Formula for this is (5000000/baud)-1, but we saturate
	   at 12 bits, which means we can't actually do anything less
	   that 1200 baud */
	switch (cflag & CBAUD) {
	case B200:	
	case B300:	
	case B1200:	clk_divisor = 4095;		break;
	case B1800:	clk_divisor = 2776;		break;
	case B2400:	clk_divisor = 2082;		break;
	case B4800:	clk_divisor = 1040;		break;
	default:
	case B9600:	clk_divisor = 519;		break;
	case B19200:	clk_divisor = 259;		break;
	case B38400:	clk_divisor = 129;		break;
	case B57600:	clk_divisor = 85;		break;
	case B115200:	clk_divisor = 42;		break;
	}
	spin_lock_irqsave(&up->port.lock, flags);
	WRITE_SERCSR(mode_reg1, up->mode_1, line);
	WRITE_SERCSR(mode_reg2, up->mode_2, line);
	WRITE_SERCSR(clk_divisor, up->clk_sel, line);
	up->last_cflags = cflag;
	spin_unlock_irqrestore(&up->port.lock, flags);
}


/* Handle notification of a termios change.  */
static void duart_set_termios(struct uart_port *port, struct termios *termios,
	struct termios *old)
{
#ifdef DUART_SPEW 
	printk("duart_set_termios called by %i (%s)\n", current->pid, current->comm);
#endif
	if (old && termios->c_cflag == old->c_cflag)
		return;
	duart_set_cflag(port->line, termios->c_cflag);
}

static void duart_stop_rx(struct uart_port *port)
{
	/* Stop accepting input */
	duart_mask_ints(port->line, M_DUART_IMR_RX);
}

static void duart_start_tx(struct uart_port *port, unsigned int tty_start)
{
	/* start transmitting */
	duart_unmask_ints(port->line, M_DUART_IMR_TX);
}

static void duart_stop_tx(struct uart_port *port, unsigned int tty_stop)
{
	/* stop transmitting */
	duart_mask_ints(port->line, M_DUART_IMR_TX);
}

static void duart_enable_ms(struct uart_port *port)
{}

static unsigned int duart_get_mctrl(struct uart_port *port)
{
	return (0);
}

static void duart_set_mctrl(struct uart_port *port, unsigned int mctrl)
{}

static void duart_break_ctl(struct uart_port *port, int break_state)
{}

static unsigned int duart_tx_empty(struct uart_port *port)
{
	duart_port_t *up = (duart_port_t *)port;
	
	return ((READ_SERCSR(up->status, port->line) & M_DUART_TX_EMT) ?
	    TIOCSER_TEMT : 0);
}

/*
 * Close a reference count out.  If reference count hits zero, null the
 * tty, kill the interrupts.  The tty_io driver is responsible for making
 * sure we've cleared out our internal buffers before calling close()
 */
static void duart_shutdown(struct uart_port *port)
{
	duart_port_t *up = (duart_port_t *)port;

	/* Stop accepting input */
	duart_mask_ints(port->line, M_DUART_IMR_RX);
	/* Wait for TX FIFO to drain */
	while (!(READ_SERCSR(up->status, port->line) & M_DUART_TX_EMT))
		;
	/* disable transmit and receive. */
	WRITE_SERCSR(M_DUART_RX_DIS|M_DUART_TX_DIS, up->cmd, port->line);

	/* Free the interrupt. */
	free_irq(port->irq, port);
}

static int __init duart_startup(struct uart_port *port) 
{
	init_duart_port((duart_port_t *)port, port->line);
	duart_mask_ints(port->line, M_DUART_IMR_ALL);
	if (request_irq(port->irq, duart_int, 0, "uart", port)) {
		panic("Couldn't get uart%d interrupt line", port->line);
	}
	/* Enable RX/TX */
	out64(M_DUART_RX_EN|M_DUART_TX_EN,
	    IO_SPACE_BASE | A_DUART_CHANREG(port->line, R_DUART_CMD));
	duart_set_cflag(port->line, DEFAULT_CFLAGS);
	/* Interrupts are now active, our ISR can be called. */
	return 0;
}

static int duart_request_port(struct uart_port *port)
{
	return (0);
}

static void duart_release_port(struct uart_port *port)
{}

static void duart_config_port(struct uart_port *port, int flags)
{
	if (flags & UART_CONFIG_TYPE)
		port->type = PORT_SB1250;
}
	
static int duart_verify_port(struct uart_port *port, struct serial_struct *ser)
{
	if (ser->type != PORT_SB1250)
		return (-EINVAL);
	if (ser->irq < 0 || ser->irq >= NR_IRQS)
		return (-EINVAL);
	return (0);
}

static const char *duart_type(struct uart_port *port)
{
	return ("sb1250_duart");
}

static struct uart_ops duart_ops = {
	.tx_empty	= duart_tx_empty,
	.set_mctrl	= duart_set_mctrl,
	.get_mctrl	= duart_get_mctrl,
	.stop_tx	= duart_stop_tx,
	.start_tx	= duart_start_tx,
	.stop_rx	= duart_stop_rx,
	.enable_ms	= duart_enable_ms,
	.break_ctl	= duart_break_ctl,
	.startup	= duart_startup,
	.shutdown	= duart_shutdown,
	.set_termios	= duart_set_termios,
	.type		= duart_type,
	.release_port	= duart_release_port,
	.request_port	= duart_request_port,
	.config_port	= duart_config_port,
	.verify_port	= duart_verify_port,
};

#ifdef CONFIG_SIBYTE_SB1250_DUART_CONSOLE
static struct console sb1250_ser_cons;
#endif

static struct uart_driver duart_reg = {
	.owner		= THIS_MODULE,
	.driver_name	= "sb1250_duart",
	.devfs_name	= "tts/",
	.dev_name	= "ttyS",
	.major		= TTY_MAJOR,
	.minor		= SB1250_DUART_MINOR_BASE,
	.nr		= DUART_MAX_LINE,
#ifdef CONFIG_SIBYTE_SB1250_DUART_CONSOLE
	.cons		= &sb1250_ser_cons,
#endif
};

static void duart_uart_init(duart_port_t *up, int line)
{
	up->port.line = line;
	up->port.flags = ASYNC_BOOT_AUTOCONF;
	up->port.fifosize = 16;
	up->port.irq = K_INT_UART_0 + line;
	up->port.ops = &duart_ops;
}

static int duart_register_ports(struct uart_driver *drv)
{
	int i;

	for (i = 0; i < DUART_MAX_LINE; i++) {
		duart_uart_init(&duart_port[i], i);
		uart_add_one_port(drv, &duart_port[i].port);
	}
	return (0);
}

static int __init sb1250_duart_init(void)
{
	int r;

printk("sb1250_duart_init: called\n");
	r = uart_register_driver(&duart_reg);
printk("sb1250_duart_init: r = %d\n", r);
	if (r == 0) {
		r = duart_register_ports(&duart_reg);
		if (r)
			uart_unregister_driver(&duart_reg);
	}
	return (r);
}

/* Unload the driver.  Unregister stuff, get ready to go away */
static void __exit sb1250_duart_fini(void)
{
	int i;

	for (i = 0; i < DUART_MAX_LINE; i++)
		uart_remove_one_port(&duart_reg, &duart_port[i].port);
	uart_unregister_driver(&duart_reg);
}

module_init(sb1250_duart_init);
module_exit(sb1250_duart_fini);
MODULE_DESCRIPTION("SB1250 Duart serial driver");
MODULE_AUTHOR("Justin Carlson, Broadcom Corp.");

#ifdef CONFIG_SIBYTE_SB1250_DUART_CONSOLE

/*
 * Serial console stuff.  Very basic, polling driver for doing serial
 * console output.  The console_sem is held by the caller, so we
 * shouldn't be interrupted for more console activity.
 * XXXKW What about getting interrupted by uart driver activity?
 */

void serial_outc(unsigned char c, int line)
{
	duart_port_t *up = duart_port + line;
	while (!(READ_SERCSR(up->status, line) & M_DUART_TX_RDY)) ;
	WRITE_SERCSR(c, up->tx_hold, line);
	while (!(READ_SERCSR(up->status, line) & M_DUART_TX_EMT)) ;
}

static void ser_console_write(struct console *cons, const char *s,
	unsigned int count)
{
	int line = cons->index;
	duart_port_t *up = duart_port + line;
	u32 imr;

	imr = READ_SERCSR(up->imr, line);
	WRITE_SERCSR(0, up->imr, line);
	while (count--) {
		if (*s == '\n')
			serial_outc('\r', line);
		serial_outc(*s++, line);
    	}
	WRITE_SERCSR(imr, up->imr, line);
}

static int __init ser_console_setup(struct console *cons, char *options)
{
	duart_port_t *up;
	int baud = 115200;
	int bits = 8;
	int parity = 'n';
	int flow = 'n';

	if (cons->index >= DUART_MAX_LINE)
		cons->index = 0;

	up = &duart_port[cons->index];
	duart_uart_init(up, cons->index);
	init_duart_port(up, cons->index);

#if SIBYTE_1956_WAR
	last_mode1[0] = V_DUART_PARITY_MODE_NONE|V_DUART_BITS_PER_CHAR_8;
#endif
#ifdef OUT
	WRITE_SERCSR(V_DUART_PARITY_MODE_NONE|V_DUART_BITS_PER_CHAR_8,
		     up->mode_1, 0);
	WRITE_SERCSR(M_DUART_STOP_BIT_LEN_1,
		     up->mode_2, 0);
	WRITE_SERCSR(V_DUART_BAUD_RATE(115200),
		     up->clk_sel, 0);
#endif
	uart_set_options(&up->port, cons, baud, parity, bits, flow);
	WRITE_SERCSR(M_DUART_RX_EN|M_DUART_TX_EN,
		     up->cmd, 0);
	return (0);
}

static struct console sb1250_ser_cons = {
	.name 		= "duart",
	.write		= ser_console_write,
	.device		= uart_console_device,
	.setup		= ser_console_setup,
	.flags		= CON_PRINTBUFFER,
	.index		= -1,
	.data		= &duart_reg
};

static int __init sb1250_serial_console_init(void)
{
	duart_reg.cons = &sb1250_ser_cons;
	register_console(&sb1250_ser_cons);
	return (0);
}

console_initcall(sb1250_serial_console_init);

#endif /* CONFIG_SIBYTE_SB1250_DUART_CONSOLE */
