/*
 * Copyright (C) 2005, Onstor, Inc.
 */
#include <linux/config.h>
#include <linux/bcd.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/mm.h>
#include <linux/bootmem.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/swap.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/timex.h>
#include <linux/vmalloc.h>
#include <asm/time.h>
#include <asm/bootinfo.h>
#include <asm/page.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/processor.h>
#include <asm/ptrace.h>
#include <asm/reboot.h>
#include <asm/tlbflush.h>
#include <asm/mipsregs.h>
#include <asm/marvell.h>
#include <linux/mv643xx.h>
#include "mv64x40.h"
#include "ds1511.h"

#define ALIGN_32M_MASK		(0x20 << 20) - 1)
#define PM_32M_ALIGNED(x)	((x) & ALIGN_32M_MASK)
#define CP0_S1_INTCONTROL_TE	(1<<7)

extern void marvell_mem_wire(void);
extern void ds1511_init_wdtimer(int sec, int msec, int resetflag);
extern unsigned long ds1511_get_time(void);
extern int ds1511_set_time(unsigned long);
extern void pcibios_set_busno_base(int busno);

/* These functions are used for rebooting or halting the machine*/
extern void onstor_bobcat_restart(char *command);
extern void onstor_bobcat_halt(void);
extern void onstor_bobcat_power_off(void);

void onstor_time_init(void);

unsigned long eee_packet_base = 0;

/*
 * The marvell register base is set to a KSEG1 address.
 * This eliminates the need for a TLB entry.
 */
unsigned long marvell_base	= MV64x40_BASE_ADDR;

EXPORT_SYMBOL(marvell_base);

static __init int per_cpu_mappings(void)
{
	int err = 0;

prom_printf("per_cpu_mappings.\n");
	write_c0_wired(0);
	local_flush_tlb_all();
	/* wire 32MB of marvell memory. */
	marvell_mem_wire();
	/* reserve a chunk of marvell memory for EEE packet driver. */
	eee_packet_base = marvel_mem_alloc(8 * (1<<20) /* 8MB */, &err);
	if (err)
		prom_printf("failed to allocate eee_packet_base\n");
dump_tlb_wired(0);
	return (0);
}
arch_initcall(per_cpu_mappings);

/*
 * use the mips alternate timer interupt, extended int 12
 */
void onstor_timer_setup(struct irqaction *irq)
{
	setup_irq(12, irq);
	/* set the "timer exclusive" bit to enable the alternate timer. */
	write_c0_intcontrol(read_c0_intcontrol() | CP0_S1_INTCONTROL_TE);
	
	/* enable the watchdog timer. */
	ds1511_init_wdtimer(16, 0, 1);
printk("onstor_timer_setup: enabled watchdog timer, intcontrol = %x\n", read_c0_intcontrol());
}

void onstor_time_init(void)
{
	extern unsigned long cpu_clock;

	/* initialize the ds1511 controll registers */
	ds1511_rtc_init();

	mips_hpt_frequency = cpu_clock / 2;
	board_timer_setup = onstor_timer_setup;

	rtc_get_time = ds1511_get_time;
	rtc_set_time = ds1511_set_time;
}

static uint32_t mv_io_base, mv_io_size;

static struct resource mv_pci_io_mem1_resource = {
	.name	= "MV64340 PCI1 IO MEM",
	.flags	= IORESOURCE_IO
};

static struct resource mv_pci_mem1_resource = {
	.name	= "MV64340 PCI1 MEM",
	.flags	= IORESOURCE_MEM
};

static struct mv_pci_controller mv_bus1_controller = {
	.pcic = {
		.pci_ops	= &mv_pci_ops,
		.mem_resource	= &mv_pci_mem1_resource,
		.io_resource	= &mv_pci_io_mem1_resource,
	},
	.config_addr	= MV64340_PCI_1_CONFIG_ADDR,
	.config_vreg	= MV64340_PCI_1_CONFIG_DATA_VIRTUAL_REG,
};

/*
 * Bobcat has two devices attached to PCI1 of the marvel chip
 * on the secondary bus.
 * compact flash:
 * 	pci_memory_base 0x18800000  	size 8MB (Not used)
 *	pci_io_memory_base 0x18000000	size 8MB
 * 2 National MACs (dp83816)
 * 	pci_memory_base 0x19000000 size 8MB
 *	pci_memory_base 0x19800000 size 8MB
 *	no IO memory window
 */
static __init void bobcat_pci1_init(void)
{
	uint32_t mem0_base, mem0_size;
	uint32_t io_base, io_size;

	/* remap compact flash to PCI address 0x800000. */
	MV_WRITE(MV64340_PCI_1_MEMORY2_LOW_ADDR_REMAP, 0x80);

	io_base = 0x18000000;
	io_size = 8 * (0x100000 /* 1MB */);
#ifdef OUT
	mem0_base = 0x19000000;
	mem0_size = 16 * (0x100000 /* 1MB */);
#endif
	mem0_base = 0x18800000;
	mem0_size = 24 * (0x100000 /* 1MB */);

	mv_pci_io_mem1_resource.start		= io_base;
	mv_pci_io_mem1_resource.end		= io_base + io_size - 1;
	mv_pci_mem1_resource.start		= mem0_base;
	mv_pci_mem1_resource.end		= mem0_base + mem0_size - 1;
	mv_bus1_controller.pcic.mem_offset	= 0;
	mv_bus1_controller.pcic.io_offset	= 0;

	ioport_resource.end		= io_base + io_size - 1;

	register_pci_controller(&mv_bus1_controller.pcic);

	mv_io_base = io_base;
	mv_io_size = io_size;
}

static __init int __init bobcat_pci_init(void)
{
	unsigned long io_v_base;
	uint32_t enable;

	enable = ~MV_READ(MV64340_BASE_ADDR_ENABLE);

	pcibios_set_busno_base(1);

	/*
	 * PCI1 I/O bit 14, PCI1 Mem1 bit 16, and PCI1 bit 17 must
	 * be enabled.
	 */
	if (enable & (0x1011 << 14)) 
		bobcat_pci1_init();

	if (mv_io_size) {
		io_v_base = (unsigned long) ioremap(0, mv_io_size);
		if (!io_v_base)
			panic("Could not ioremap I/O port range");

prom_printf("bobcat_pci_init:io_v_base = %x\n", io_v_base);
		set_io_port_base(io_v_base);
	}

	return (0);
}

arch_initcall(bobcat_pci_init);

int  __init plat_setup(void)
{
	board_time_init = onstor_time_init;

	_machine_restart = onstor_bobcat_restart;
	_machine_halt = onstor_bobcat_halt;
	_machine_power_off = onstor_bobcat_power_off;

	/*
	 * initrd_start = (ulong)jaguar_initrd_start;
	 * initrd_end = (ulong)jaguar_initrd_start + (ulong)jaguar_initrd_size;
	 * initrd_below_start_ok = 1;
	 */


#ifdef BOBCAT_SMP
	/* 512MB */
	add_memory_region(0x0, 0x200<<20, BOOT_MEM_RAM);
#else
	/* 256MB */
	add_memory_region(0x0, 0x100<<20, BOOT_MEM_RAM);
#endif
	return (0);
}
